"""Settings page for configuring reminders and viewing today's email logs."""

from __future__ import annotations

from datetime import date
from typing import Any, Dict, List

import pandas as pd
import streamlit as st

from services.settings_service import SettingsService
from services.email_service import EmailService

# Repos for reading logs & employee names
from repositories.email_log_repo import EmailLogRepository
from repositories.employee_repo import EmployeeRepository


def _get_all(repo: Any) -> List[Dict[str, Any]]:
    """Repo-agnostic fetch: list_all() -> get_all() -> all() -> table.all()."""
    for name in ("list_all", "get_all", "all"):
        if hasattr(repo, name):
            rows = getattr(repo, name)()
            return rows if isinstance(rows, list) else list(rows)
    if hasattr(repo, "table"):
        return [dict(r) for r in repo.table.all()]
    return []


def _today_logs(repo: EmailLogRepository) -> List[Dict[str, Any]]:
    """Fetch email logs for today; prefer repo API, else filter all rows by date prefix."""
    if hasattr(repo, "logs_on_date"):
        try:
            return repo.logs_on_date(date.today())  # type: ignore[return-value]
        except Exception:
            pass
    # Fallback: filter by sent_at date prefix
    rows = _get_all(repo)
    prefix = date.today().isoformat()
    out: List[Dict[str, Any]] = []
    for r in rows:
        s = str(r.get("sent_at", ""))
        if s.startswith(prefix):
            out.append(r)
    return out


def render() -> None:
    st.title("Settings")

    role = str(st.session_state.get("role", ""))  # Admin | Manager | Staff
    if role not in ("Admin", "Manager"):
        st.info("You have view-only access to Settings.")
        return

    settings_service = SettingsService()
    email_service = EmailService()
    log_repo = EmailLogRepository()
    emp_repo = EmployeeRepository()

    # -------- Current settings --------
    raw = settings_service.get_settings()
    if isinstance(raw, dict):
        look_ahead_days = int(raw.get("look_ahead_days", 7))
        timezone = str(raw.get("timezone", "Asia/Kolkata"))
    else:
        look_ahead_days = int(getattr(raw, "look_ahead_days", 7))
        timezone = str(getattr(raw, "timezone", "Asia/Kolkata"))

    with st.form("settings_form"):
        new_lookahead = st.number_input(
            "Reminder Look-ahead (days)", min_value=1, max_value=60, value=look_ahead_days, step=1, key="set_lookahead"
        )
        new_timezone = st.text_input("Timezone (IANA)", value=timezone, key="set_tz")
        saved = st.form_submit_button("Save Settings")
    if saved:
        try:
            # Accept dict or model; SettingsService handles persistence
            settings_service.update_settings({"look_ahead_days": int(new_lookahead), "timezone": new_timezone})
            st.success("Settings saved.")
            st.rerun()
        except Exception as e:
            st.error(f"Could not save settings: {e}")

    # -------- Send reminders now --------
    if st.button("Send Reminders Now", key="send_now_btn"):
        try:
            result = email_service.send_due_reminders(look_ahead_days)
            st.success(
                f"Reminders processed: attempted {result.get('emails_attempted', 0)}, "
                f"sent {result.get('emails_sent', 0)}, "
                f"skipped/failed {result.get('emails_skipped_or_failed', 0)}."
            )
            # Immediately load today's logs and show a table of SENT emails
            logs = _today_logs(log_repo)

            # Join employee names
            emp_map = {int(e["id"]): e.get("full_name", "") for e in _get_all(emp_repo) if "id" in e}
            for r in logs:
                try:
                    r["employee_name"] = emp_map.get(int(r.get("employee_id") or 0), "")
                except Exception:
                    r["employee_name"] = ""

            df = pd.DataFrame(logs) if logs else pd.DataFrame(
                columns=["employee_id", "employee_name", "email", "subject", "status", "sent_at"]
            )

            # Filter to only the emails actually SENT
            if not df.empty:
                df["sent_at"] = pd.to_datetime(df["sent_at"], errors="coerce")
                sent_df = df[df["status"].astype(str).str.upper().str.startswith("SENT")].copy()
                sent_df = sent_df[
                    ["employee_id", "employee_name", "email", "subject", "status", "sent_at"]
                ].sort_values(by="sent_at", ascending=False)
            else:
                sent_df = df

            st.subheader("Today's Email Reminders (Sent)")
            if sent_df.empty:
                st.info("No emails sent yet today.")
            else:
                st.dataframe(sent_df, use_container_width=True)
                csv_bytes = sent_df.to_csv(index=False).encode("utf-8")
                st.download_button(
                    "Download Sent Emails (CSV)",
                    data=csv_bytes,
                    file_name="email_reminders_sent_today.csv",
                    mime="text/csv",
                    key="dl_sent_emails_csv",
                )
        except Exception as e:
            st.error(f"Failed to send reminders: {e}")

    # -------- Always show today's logs summary (optional) --------
    st.markdown("—")
    with st.expander("View all email reminder attempts today"):
        logs = _today_logs(log_repo)
        if logs:
            df_all = pd.DataFrame(logs)
            df_all["sent_at"] = pd.to_datetime(df_all["sent_at"], errors="coerce")
            st.dataframe(df_all.sort_values(by="sent_at", ascending=False), use_container_width=True)
        else:
            st.write("No reminder attempts logged today.")
